// UV Stepper.js
//	this is Tool script, place it into ~/Library/Application Support/Cheetah3D/scripts/Tool folder
//
//	for setting or rotating UV Coordinates with certain value, and for alignment.
//
//  (c) 2006-2014 Hiroto Tsubaki
//  http://www.tres-graficos.jp/
//  tg@tres-graficos.jp
//
// 2008-02-17: update. now added Point/Edge mode / added 'none' for alignment value / fixed center, middle alignment bug
// 2008-05-26: bug fix for 4.6 update
// 2008-07-16: add new function swapUV
// 2008-12-13: add new function Weld
// 2008-12-16: add new function Invert / parameter order changed
// 2009-03-11: add new function Copy object UV. a behavior of collection for selection changed.
// 2009-03-11: add error check when use this with befroe 3.6 version
// 2009-06-19: add UV pin select function.
// 2009-06-20: add align circle function.
// 2010-01-25: fix bug for 5.2;
// 2012-04-11: add getAngle function, undoable for v6.0
// 2014-04-19: circle alignment bug fixed.

// add function to Array class.
if( !Array.indexOf) {
	Array.prototype.indexOf = function(str) {
		for(var i = 0; i < this.length; i++) {
			if(this[i] == str)
				return i;
		}
		return -1;
	}
}
if (!Array.isExists) {
	Array.prototype.isExists = function(value) {
		for (var i = 0;i < this.length;i++) {
			if (this[i] == value) {
				return true;
			}
		}
		return false;
	}
}

function buildUI(tool) {
	tool.addParameterSeparator("UV Stepper");
	
	tool.addParameterBool("use u factor",1,0,1,true,true);
	tool.addParameterFloat("u factor",0,-10,10,true,true);
	tool.addParameterBool("use v factor",1,0,1,true,true);
	tool.addParameterFloat("v factor",0,-10,10,true,true);
	tool.addParameterFloat("rotation",0,-180,180,true,true);
	tool.addParameterSelector("coordinate",["selected center","0.5,0.5","0.0,0.0"],true,true);
	
	tool.addParameterButton("move","move","Move");
	tool.addParameterButton("rotate","rotate","Rotate");
	tool.addParameterButton("scale","scale","Scale");
	
	tool.addParameterButton("move to", "move", "MoveTo");
	
	tool.addParameterSeparator("Set Parameter");
	tool.addParameterButton("reset","reset","Reset");
	tool.addParameterButton("invert","invert","Invert");
	
  tool.addParameterSeparator("Get Angle");
  tool.addParameterBool("auto set angle", 1, 0, 1, false, false);
  tool.addParameterButton("get angle", "get", "getAngle");
  
  tool.addParameterSeparator("Get Fit Scale");
  tool.addParameterBool("auto set scale", 1, 0, 1, false, false);
  tool.addParameterButton("get scale", "get", "getFitScale");
  
	tool.addParameterSeparator("Invert");
	tool.addParameterButton("invert u","invert","doInvertU");
	tool.addParameterButton("invert v","invert","doInvertV");
	
	tool.addParameterSeparator("Point Alignment");
	tool.addParameterButton("alignment h","align","doPointAlignH");
	tool.addParameterButton("alignment v","align","doPointAlignV");
	tool.addParameterFloat("circle radius", 0, 0, 1, false, false);
	tool.addParameterButton("alignment circle", "align", "doPointAlignCircle");
	
	tool.addParameterSeparator("Stage Alignment");
	tool.addParameterSelector("align u",["none","left","center","right"],true,true);
	tool.addParameterSelector("align v",["none","top","middle","bottom"],true,true);
	tool.addParameterButton("alignment","align","doAlign");
	
	tool.addParameterSeparator("Weld UV");
	tool.addParameterButton("weld","weld","WeldUV");
	
	tool.addParameterSeparator("Selection +");
	tool.addParameterButton("select UV pin", "select", "selectUVPin");
	
	tool.addParameterSeparator("Copy UV set");
	tool.addParameterSelector("copy type", ["UV1 to UV2", "UV2 to UV1", "swap UV1, UV2"], false, false);
	tool.addParameterButton("copy","copy","CopyUV");
	
	tool.addParameterSeparator("Copy UV set to Object");
	tool.addParameterButton("copy", "copy", "CopyStoredUV");
	tool.addParameterButton("paste", "paste", "PasteStoredUV");
	/*
	tool.addParameterSeparator("Set");
	tool.addParameterBool("use u value",1,0,1,true,true);
	tool.addParameterFloat("u value",0,-10,10,true,true);
	tool.addParameterBool("use v value",1,0,1,true,true);
	tool.addParameterFloat("v value",0,-10,10,true,true);
	tool.addParameterButton("set","set","Set");
	*/
}

function Reset(tool) {
  if (tool.parameterWithName) tool.recordParametersForUndo(); // for Undoing
  
	tool.setParameter("u factor",0);
	tool.setParameter("v factor",0);
	tool.setParameter("rotation",0);
}

function Invert(tool) {
  if (tool.parameterWithName) tool.recordParametersForUndo(); // for Undoing
  
	tool.setParameter("u factor",-1 * tool.getParameter("u factor"));
	tool.setParameter("v factor",-1 * tool.getParameter("v factor"));
	tool.setParameter("rotation",-1 * tool.getParameter("rotation"));
}

function setUVWithList(core, pList, uvList) {
	var i;
	var count = pList.length;
	//print(uvList.length+':'+uvList);
	var pc = core.polygonCount();
	var ps = 0;
	for (i = 0;i < count;i++) {
		if (pc > pList[i][0]) {
			ps = core.polygonSize(pList[i][0]);
			if (ps > pList[i][1]) core.setUVCoord(pList[i][0], pList[i][1], uvList[i]);
		}
	}
}

function collectSelection(core, mode) {
	var i, j;
	var polyCount = core.polygonCount();
	var vList = new Array;
	var uvList = new Array;
	switch(mode) {
		case POLY_MODE:
			for (i = 0;i < polyCount;i++) {
				if (core.polygonSelection(i)) {
					var polySize = core.polygonSize(i);
					for (j = 0;j < polySize;j++) {
						var uv = core.uvCoord(i,j);
						uvList.push(uv);
						vList.push([i,j]);
					}
				}
			}
			break;
		case EDGE_MODE:
		case POINT_MODE:
			if (mode == EDGE_MODE) {
				var detectMode = UVEDGE;
			} else {
				var detectMode = UVPOINT;
			}
			for (i = 0;i < polyCount;i++) {
				var polySize = core.polygonSize(i);
				for (j = 0;j < polySize;j++) {
					if (core.edgeSelection(i,j,detectMode)) {
						var uv = core.uvCoord(i,j);
						uvList.push(uv);
						vList.push([i,j]);
						if (detectMode == UVEDGE) {
							var k = (j == polySize - 1)? 0 : j+1;
							//print('adding k:'+k+' for edge:'+j);
							uv = core.uvCoord(i,k);
							uvList.push(uv);
							vList.push([i,k]);
						}
					}
				}
			}
			break;
		case OBJECT_MODE:
			OS.messageBox("Point/Edge/Polygon Mode only","This tool works with just Point/Edge/Polygon Mode.");
			break;
	}
	if (uvList.length == 0) { // if no selection, recollect all info of uvs.
		for (i = 0;i < polyCount;i++) {
			var polySize = core.polygonSize(i);
			for (j = 0;j < polySize;j++) {
				var uv = core.uvCoord(i,j);
				uvList.push(uv);
				vList.push([i,j]);
			}
		}
	}
	return [vList, uvList];
}

function moveWithMode(tool, mode, moveMode) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (obj && obj.type() == POLYGONOBJ) {
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
		
    var core = obj.core();
		var i,j;
		var coor = parseInt(tool.getParameter("coordinate"));
		var set = obj.getParameter("activeuvset");
		var u = (tool.getParameter("use u factor"))? tool.getParameter("u factor"):0;
		var v = (tool.getParameter("use v factor"))? tool.getParameter("v factor"):0;
		if (mode == SCALE) {
			if (!tool.getParameter("use u factor")) var u = 1;
			if (!tool.getParameter("use v factor")) var v = 1;
		}
		var rotation = tool.getParameter("rotation");
		if (mode == ROTATE) {
			var mat = new Mat3D(mode, rotation);
		} else {
			var mat = new Mat3D(mode, u, v);
		}
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
		var vec4d = uvList[1].hasOwnProperty("w");
		
		//print(uvList);
		if (vList.length > 0) {
			var len = uvList.length;
			centerVec = getCenterVecForSelector(coor, uvList, set);
			//print(centerVec);
			for (i = 0;i < len;i++) {
				var uvR = uvList[i];
				if (mode == TRANSLATE) {
					if (moveMode) {
						var transVec = new Vec2D(u,v);
						transVec = transVec.sub( centerVec ); // move distance
						if (tool.getParameter("use u factor") && tool.getParameter("use v factor")) {
							mat = new Mat3D( TRANSLATE, transVec.x, transVec.y );
						} else if (tool.getParameter("use u factor")) {
							mat = new Mat3D( TRANSLATE, transVec.x, 0 );
						} else if (tool.getParameter("use v factor")) {
							mat = new Mat3D( TRANSLATE, 0, transVec.y );
						}
					}
					var uvH = (set == 0 || ! vec4d)? new Vec2D(uvR.x, uvR.y) : new Vec2D(uvR.z, uvR.w);
					uvH = mat.multiply(uvH);
					uvR = (vec4d)? 
							(set == 0)?
								new Vec4D(uvH.x, uvH.y, uvR.z, uvR.w) :
								new Vec4D(uvR.x, uvR.y, uvH.x, uvH.y)  :
							new Vec2D(uvH.x, uvH.y);
				} else {
					var uvH = (set == 0 || ! vec4d)? new Vec2D(uvR.x, uvR.y) : new Vec2D(uvR.z, uvR.w);
					uvH = uvH.sub(centerVec);
					uvH = mat.multiply(uvH);
					uvH = uvH.add(centerVec);
					uvR = (vec4d)? 
							(set == 0)?
								new Vec4D(uvH.x, uvH.y, uvR.z, uvR.w) :
								new Vec4D(uvR.x, uvR.y, uvH.x, uvH.y)  :
							new Vec2D(uvH.x, uvH.y);
				}
				uvList[i] = uvR;
			}
			//print(uvList);
			setUVWithList(core, vList, uvList);
		}
		obj.update();
	}
}

function MoveTo(tool) {
	moveWithMode(tool, TRANSLATE, true);
}

function Move(tool) {
	moveWithMode(tool, TRANSLATE, false);
}

function Rotate(tool) {
	moveWithMode(tool, ROTATE, false);
}

function Scale(tool) {
	moveWithMode(tool, SCALE, false);
}

function doInvertU(tool) {
	var u_s = tool.getParameter("use u factor");
	var v_s = tool.getParameter("use v factor");
	var u_v = tool.getParameter("u factor");
	var v_v = tool.getParameter("v factor");
	var cor = parseInt(tool.getParameter("coordinate"));

  tool.setParameter("use u factor", 1);
	tool.setParameter("use v factor", 1);
	tool.setParameter("u factor", -1);
	tool.setParameter("v factor", 1);
	tool.setParameter("coordinate", 0);
	
	moveWithMode(tool, SCALE, false);

	tool.setParameter("use u factor", u_s);
	tool.setParameter("use v factor", v_s);
	tool.setParameter("u factor", u_v);
	tool.setParameter("v factor", v_v);
	tool.setParameter("coordinate", cor);
}

function doInvertV(tool) {
	var u_s = tool.getParameter("use u factor");
	var v_s = tool.getParameter("use v factor");
	var u_v = tool.getParameter("u factor");
	var v_v = tool.getParameter("v factor");
	var cor = parseInt(tool.getParameter("coordinate"));

	tool.setParameter("use u factor", 1);
	tool.setParameter("use v factor", 1);
	tool.setParameter("u factor", 1);
	tool.setParameter("v factor", -1);
	tool.setParameter("coordinate", 0);
	
	moveWithMode(tool, SCALE, false);

	tool.setParameter("use u factor", u_s);
	tool.setParameter("use v factor", v_s);
	tool.setParameter("u factor", u_v);
	tool.setParameter("v factor", v_v);
	tool.setParameter("coordinate", cor);
}

function getCenterVecForSelector(coor, list, set) {
	var i;
	var len = list.length;
	var centerVec = new Vec2D(0,0);
	var vec4d = list[0].hasOwnProperty("w");
	
	switch (coor) {
		case 0: // selected
			for (i = 0; i < len;i++) {
				var vec = (set == 0 || !vec4d)?
						new Vec2D(list[i].x, list[i].y) :
						new Vec2D(list[i].z, list[i].w);
				centerVec = centerVec.add(vec);
			}
			centerVec = centerVec.multiply(1/len);
			break;
		case 1: // uv 0.5, 0.5
			centerVec = new Vec2D(0.5, 0.5);
			break;
		case 2: // uv 0, 0
			break;
	}
	return centerVec;
}

function getMoveVecForAlign(au, av, list, set) { // 0/1/2/3 - none/left/center/right, 0/1/2/3 - none/top/middle/bottom
	var i;
	var len = list.length;
	
	var cmVec = new Vec2D(0.5,0.5);
	var pVec = new Vec2D(0, 0);
	var minVec = new Vec2D(100000, 100000);
	var maxVec = new Vec2D(-100000, -100000);
	var vec4d = list[1].hasOwnProperty("w");
	
	for (i = 0;i < len;i++) {
		var vec = (set == 0 || !vec4d)?
					new Vec2D(list[i].x, list[i].y) :
					new Vec2D(list[i].z, list[i].w) ;
		pVec = pVec.add(vec);
		minVec.x = (vec.x < minVec.x)? vec.x : minVec.x;
		minVec.y = (vec.y < minVec.y)? vec.y : minVec.y;
		maxVec.x = (vec.x > maxVec.x)? vec.x : maxVec.x;
		maxVec.y = (vec.y > maxVec.y)? vec.y : maxVec.y;
	}
	pVec = pVec.multiply(1/len);
	
	if (au == 2) pVec.x = minVec.x + (maxVec.x - minVec.x)/2;
	if (av == 2) pVec.y = minVec.y + (maxVec.y - minVec.y)/2;
	
	/*
	print('---- check alignment value');
	print('pVec -b:'+pVec);
	*/
	switch(au) {
		case 0: // none
			pVec.x = 0;
			break;
		case 1: // left
			pVec.x = -1 * minVec.x;
			break;
		case 2: // center
			pVec.x = cmVec.x - pVec.x;
			break;
		case 3: // right
			pVec.x = 1 - maxVec.x;
			break;
	}
	switch(av) {
		case 0: // none
			pVec.y = 0;
			break;
		case 1: // top
			pVec.y = -1 * minVec.y;
			break;
		case 2: // middle
			pVec.y = cmVec.y - pVec.y;
			break;
		case 3: // bottom
			pVec.y = 1 - maxVec.y;
			break;
	}
	/*
	print('min - '+minVec);
	print('max - '+maxVec);
	print('pVec -a:'+pVec);
	print('---- check alignment value');
	*/
	
	return pVec;
}

function doPointAlignH(tool) {
	doPointAlign(tool, 0);
}
function doPointAlignV(tool) {
	doPointAlign(tool, 1);
}

function doPointAlign(tool, mode) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (doc.editMode() != POINT_MODE && doc.editMode() != EDGE_MODE) { // weld work on only Point mode / Edge mode.
		OS.beep();
		return;
	}
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i,j;
		
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];

		if (vList.length > 0) {
			var vec4d = uvList[0].hasOwnProperty("w");
			
			var len = uvList.length;
			var pVec = new Vec2D();
			for (i = 0;i < len;i++) {
				var vec = (set == 0 || !vec4d)?
							new Vec2D(uvList[i].x, uvList[i].y) :
							new Vec2D(uvList[i].z, uvList[i].w);
				pVec = pVec.add(vec);
			}
			pVec = pVec.multiply(1/len);
			for (i = 0;i < len;i++) {
				var uvR = uvList[i];
				if (mode == 1) { // vertical
					uvList[i] = (vec4d)? 
								(set == 0)? 
									new Vec4D(pVec.x, uvR.y, uvR.z, uvR.w) :
									new Vec4D(uvR.x, uvR.y, pVec.x, uvR.w)  :
								new Vec2D(pVec.x, uvR.y);
				} else { // horizontal
					uvList[i] = (vec4d)? 
								(set == 0)? 
									new Vec4D(uvR.x, pVec.y, uvR.z, uvR.w) :
									new Vec4D(uvR.x, uvR.y, uvR.z, pVec.y)  :
								new Vec2D(uvR.x, pVec.y);
				}
			}
			setUVWithList(core, vList, uvList);
		}
		obj.update();
	}
}

function Vec2D_distance( v1, v2 ) {
  return Math.sqrt( v1.x*v2.x + v1.y*v2.y );
}

function doPointAlignCircle( tool ) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (doc.editMode() != POINT_MODE) {
		OS.beep(); return;
	}
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i, j;
		
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
		
		if (vList.length > 0) {
			var vec4d = uvList[0].hasOwnProperty("w");
			
			var len = uvList.length;
			var vec_cache = [];
			var pVec = new Vec2D();
			for (i = 0;i < len;i++) {
				var vec = (set == 0 || !vec4d)?
								new Vec2D(uvList[i].x, uvList[i].y) :
								new Vec2D(uvList[i].z, uvList[i].w) ;
				vec_cache.push( vec.copy() );
				pVec = pVec.add(vec);
			}
			var centerVec = pVec.multiply(1/len);
			//
			//var radius = vec_cache[0].len(centerVec);
			var radius = (tool.getParameter("circle radius"))? tool.getParameter("circle radius") : Vec2D_distance( vec_cache[0], centerVec );
			
			//print('center:'+centerVec+', radius:'+radius);
			
			var space = [];
			space[0] = [];
			space[1] = [];
			
			for (i = 0;i < len;i++) {
				var vec = vec_cache[i].sub(centerVec);
				var s = Math.atan2( vec.y, vec.x );
				
				if (s >= 0) { // 0 - 180
					var new_list = [];
					var pushed = false;
					for (j = 0;j < space[0].length;j++) {
						var old_vec = space[0][j];
						if (old_vec[0] < s) {
							new_list.push(old_vec);
						} else if (old_vec[0] == s) {
							var comp_vec = vec_cache[ old_vec[1][0] ].sub(centerVec);
							if (comp_vec.x == vec.x && comp_vec.y == vec.y) {
								if (pushed == false) {
									old_vec[1].push( i );
									pushed = true;
								}
								new_list.push( old_vec );
							} else {
								if (pushed == false) {
									new_list.push( [ s, [i] ] );
									pushed = true;
								}
								new_list.push( old_vec );
							}
						} else {
							if (pushed == false) {
								new_list.push( [ s, [ i ] ] );
								pushed = true;
							}
							new_list.push( old_vec );
						}
					}
					if (pushed == false) {
						new_list.push( [ s, [ i ] ] );
					}
					space[0] = new_list;
				} else {
					var new_list = [];
					var pushed = false;
					for (j = 0;j < space[1].length;j++) {
						var old_vec = space[1][j];
						if (old_vec[0] < s) {
							new_list.push(old_vec);
						} else if (old_vec[0] == s) {
							var comp_vec = vec_cache[ old_vec[1][0] ].sub(centerVec);
							if (comp_vec.x == vec.x && comp_vec.y == vec.y) {
								if (pushed == false) {
									old_vec[1].push( i );
									pushed = true;
								}
								new_list.push( old_vec );
							} else {
								if (pushed == false) {
									new_list.push( [ s, [i] ] );
									pushed = true;
								}
								new_list.push( old_vec );
							}
						} else {
							if (pushed == false) {
								new_list.push( [ s, [ i ] ] );
								pushed = true;
							}
							new_list.push( old_vec );
						}
					}
					if (pushed == false) {
						new_list.push( [ s, [ i ] ] );
					}
					space[1] = new_list;
				}
			}
			var list = [];
			var check = [];
			for (i = 0;i < 2;i++) {
				var vals = space[i];
				for (j = 0;j < vals.length;j++) {
					list.push(vals[j][1]);
					check.push(vals[j][0]);
				}
			}
			
			/*
			for (i = 0;i < check.length;i++) {
				print(check[i].toFixed(3));
			}
			*/
			
			var count = list.length;
			var seg = ( Math.PI * 2 ) / count;
			for (i = 0;i < count;i++) {
				var cir_vec = new Vec2D(Math.cos( seg * i) * radius, Math.sin( seg * i) * radius );
				cir_vec = cir_vec.add( centerVec );
				
				for (j = 0;j < list[i].length;j++) {
					
					var uvR = uvList[ list[i][j] ];
					
					uvList[ list[i][j] ] = (vec4d)? 
										  (set == 0)? 
										  		new Vec4D(cir_vec.x, cir_vec.y, uvR.z, uvR.w) :
										  		new Vec4D(uvR.x, uvR.y, cir_vec.x, cir_vec.y) :
										  new Vec2D( cir_vec.x, cir_vec.y );
					
				}
			}
			setUVWithList(core, vList, uvList);
		}
		
		obj.update();
	}
}

function getFitScale( tool ) {
  var autoset = tool.getParameter("auto set scale");
  var doc = tool.document();
  var obj = doc.selectedObject();
  
  if (obj && obj.type() == POLYGONOBJ) {
    var core = obj.core();
    var i, j;
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
    
    if (uvList.length > 1) {
      var vec4d = uvList[0].hasOwnProperty("w");
      var vec, vecMin, vecMax;
      
      var len = uvList.length;
      vec = (vec4d)?
              (set == 0)?
                new Vec2D(uvList[0].x, uvList[0].y) :
                new Vec2D(uvList[0].z, uvList[0].w)  :
              new Vec2D(uvList[0].x, uvList[0].y);
      vecMin = vec.copy();
      vecMax = vec.copy();
      
      for (i = 1;i < len;i++) {
        vec = (vec4d)?
                (set == 0)?
                  new Vec2D(uvList[i].x, uvList[i].y) :
                  new Vec2D(uvList[i].z, uvList[i].w)  :
                new Vec2D(uvList[i].x, uvList[i].y);
        
        vecMax.x = ( vec.x < vecMax.x )? vecMax.x : vec.x;
        vecMax.y = ( vec.y < vecMax.y )? vecMax.y : vec.y;
        vecMin.x = ( vec.x > vecMin.x )? vecMin.x : vec.x;
        vecMin.y = ( vec.y > vecMin.y )? vecMin.y : vec.y;
      }
      
      var dx = vecMax.x - vecMin.x;
      var dy = vecMax.y - vecMin.y;
      
      //print( 'max:'+vecMax.x.toFixed(3)+', '+vecMax.y.toFixed(3));
      //print( 'min:'+vecMin.x.toFixed(3)+', '+vecMin.y.toFixed(3));
      var scale = (dx > dy)? dx : dy;
      
      print('fitScale:'+scale);
      
      if (autoset) {
        if (tool.parameterWithName) tool.recordParametersForUndo(); // for Undoing
        tool.setParameter("u factor", 1/scale);
        tool.setParameter("v factor", 1/scale);
      }
    }
  
  }
}

function getAngle( tool ) {
  var autoset = tool.getParameter("auto set angle");
  var doc = tool.document();
  var obj = doc.selectedObject();
  
	if (doc.editMode() != POINT_MODE && doc.editMode() != EDGE_MODE) { // weld work on only Point mode / Edge mode.
		OS.beep();
		return;
	}
  
  if (obj && obj.type() == POLYGONOBJ) {
    var core = obj.core();
    var i, j;
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
    
    if (uvList.length > 1) {
			var vec4d = uvList[0].hasOwnProperty("w");
			var vec1, vec2, vec3;
      
			var len = uvList.length;
      vec1 = (vec4d)?
              (set == 0)?
                new Vec2D(uvList[0].x, uvList[0].y) :
                new Vec2D(uvList[0].z, uvList[0].w)  :
              new Vec2D(uvList[0].x, uvList[0].y);
      vec2 = (vec4d)?
              (set == 0)?
                new Vec2D(uvList[1].x, uvList[1].y) :
                new Vec2D(uvList[1].z, uvList[1].w)  :
              new Vec2D(uvList[1].x, uvList[1].y);
      vec3 = vec1.sub(vec2);
      if (Math.sqrt( vec3.x * vec3.x + vec3.y * vec3.y ) == 0) {
        var cs = 0;
      } else {
        var cs = Math.acos( vec3.x / ( Math.sqrt( vec3.x * vec3.x + vec3.y * vec3.y) ) );
      }
      
      var angle = (cs / Math.PI) * 180;
      
      if (vec3.y < 0) {
        angle = 360 - angle;
      }
      print( 'angle:'+angle );
      
      if (angle > 180) {
        angle = angle - 180;
      }
      if (autoset) {
        if (tool.parameterWithName) tool.recordParametersForUndo(); // for Undoing
        tool.setParameter("rotation", angle * -1);
      }
    }
  }
}

function selectUVPin( tool ) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i;
		
		switch(doc.editMode()) {
			case POINT_MODE:
				var collection = collectSelection(core, doc.editMode());
				var vList = collection[0];
				var len = vList.length;
				
        if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
        
				for (i = 0;i < len;i++) {
					//if (! core.edgeSelection( vList[i][0], vList[i][1], UVPINNED )) { // bug:: UVPINNED is not defined.
					if (! core.edgeSelection( vList[i][0], vList[i][1], 16 )) {
						core.setEdgeSelection( vList[i][0], vList[i][1], UVPOINT, false);
						core.setVertexSelection( core.vertexIndex(vList[i][0], vList[i][1]), false);
					}
				}
				obj.update();
				break;
		}
	}
}

function doAlign( tool ) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	var alignU = parseInt(tool.getParameter("align u"));
	var alignV = parseInt(tool.getParameter("align v"));
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i,j;
		
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
		var vec4d = uvList[1].hasOwnProperty("w");
		//print("vec4d:"+vec4d);
		
		if (vList.length > 0) {
			moveVec = getMoveVecForAlign(alignU, alignV, uvList, set);
			var mat = new Mat3D(TRANSLATE, moveVec.x, moveVec.y);
			var len = uvList.length;
			for (i = 0;i < len;i++) {
				var uvR = uvList[i];
				var uvH = (vec4d)? 
							(set == 0)? new Vec2D(uvR.x, uvR.y) : new Vec2D(uvR.z, uvR.w) :
							new Vec2D(uvR.x, uvR.y);
							
				uvH = mat.multiply(uvH);
				uvList[i] = (vec4d)? 
							(set == 0)? 
								new Vec4D(uvH.x, uvH.y, uvR.z, uvR.w) :
								new Vec4D(uvR.x, uvR.y, uvH.x, uvH.y)  :
							new Vec2D(uvH.x, uvH.y);
			}
			setUVWithList(core, vList, uvList);
		}
		obj.update();
	}
}

function CopyUV(tool) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i,j;
		var copyType = parseInt(tool.getParameter("copy type"));
		//
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
		
		if (! uvList.length) {
			return;
		}
		
		var vec4d = uvList[1].hasOwnProperty("w");
		
		if (! vec4d) {
			OS.messageBox('Cheetah3D 4.6 above required!!', 'Copy UV function is available for Cheetah3D 4.6 above.');
			return;
		}

    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
		
		if (vList.length > 0) {
			var len = uvList.length;
			for (i = 0;i < len;i++) {
				var uvR = uvList[i];
				switch(copyType) {
					case 0: // UV1 to UV2
						uvR.z = uvR.x;
						uvR.w = uvR.y;
						break;
					case 1: // UV2 to UV1
						uvR.x = uvR.z;
						uvR.y = uvR.w;
						break;
					case 2: // swap
						uvR = new Vec4D(uvR.z, uvR.w, uvR.x, uvR.y);
						break;
				}
				uvList[i] = uvR;
			}
			setUVWithList(core, vList, uvList);
		}
		obj.update();
	}
}

var storedUV = [];
var storedSet = '';

function CopyStoredUV(tool) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i, j;
		
		storedUV = collectSelection(core, doc.editMode());
		storedSet = obj.getParameter("activeuvset");
	}
}

function PasteStoredUV(tool) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i, j;
		
		if (! storedUV.length) {
			return;
		}
		
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
    
		var set = obj.getParameter("activeuvset");
		var vList = storedUV[0];
		var uvList = storedUV[1];
		
		if (!uvList && ! uvList.length) {
			return;
		}
		
		var collection = collectSelection(core, doc.editMode());
		var ovList = collection[0];
		var ouvList = collection[1];
		var vec4d = uvList[1].hasOwnProperty("w");
		
		var resultUVList = [];
		
		if (ovList.length != vList.length) {
			var res = OS.messageBox("Object's UV information doesn't match stored UV information!!", "Do you want to paste UV info by force?");
			if (!res) return;
		}
		if (vList.length > 0) {
			var len = Math.min(vList.length, ovList.length);
			for (i = 0;i < len;i++) {
				var u = (vec4d != 4 && storedSet == 0)? uvList[i].x : uvList[i].z;
				var v = (vec4d != 4 && storedSet == 0)? uvList[i].y : uvList[i].w;
				resultUVList[i] = (vec4d)?
							(set == 0)?
								new Vec4D(u, v, ouvList[i].z, ouvList[i].w) :
								new Vec4D(ouvList[i].x, ouvList[i].y, u, v)  :
							new Vec2D(u, v);
			}
			setUVWithList(core, vList, resultUVList);
		}
		obj.update();
	}
}

function WeldUV(tool) {
	var doc = tool.document();
	var obj = doc.selectedObject();
	
	if (doc.editMode() != POINT_MODE) { // weld work on only Point mode.
		OS.beep();
		return;
	}
	
	if (obj && obj.type() == POLYGONOBJ) {
		var core = obj.core();
		var i,j;
		
    if (tool.parameterWithName) obj.recordGeometryForUndo(); // for Undoing
    
		var set = obj.getParameter("activeuvset");
		// detecting selected vertices.
		var collection = collectSelection(core, doc.editMode());
		var vList = collection[0];
		var uvList = collection[1];
		var vec4d = uvList[0].hasOwnProperty("w");
		
		var len = vList.length;
		var uv_cache = new Array;
		for (i = 0; i < len; i++) {
			var vec_index = core.vertexIndex(vList[i][0], vList[i][1]);
			var uv_len = uv_cache.length;
			var unique = true;
			for (j = 0;j < uv_len;j++) {
				var uv_c = uv_cache[j];
				if (uv_c[0] == vec_index) {
					uv_cache[j][1].push(vList[i]);
					uv_cache[j][2].push(uvList[i]);
					unique = false;
					break;
				}
			}
			if (unique) uv_cache.push([ vec_index, [ vList[i] ], [ uvList[i] ] ]);
		}
    
		v_list = new Array;
		uv_list = new Array;
		len = uv_cache.length;
		for (i = 0;i < len;i++) {
			uv_c = uv_cache[i];
			uv_len = uv_c[2].length;
			var uv_cal = new Vec2D();
			for (j = 0;j < uv_len;j++) {
				var uvR = uv_c[2][j];
				var uvH = (vec4d)?
							(set == 0)? new Vec2D(uvR.x, uvR.y) : new Vec2D(uvR.z, uvR.w) :
							new Vec2D(uvR.x, uvR.y);
				uv_cal = uv_cal.add(uvH);
			}
			uv_cal = uv_cal.multiply(1/uv_len);
			
			for (j = 0;j < uv_len;j++) {
				var uvR = uv_c[2][j];
				var uv_tol = (vec4d)?
							 (set == 0)?
								new Vec4D(uv_cal.x, uv_cal.y, uvR.z, uvR.w) :
								new Vec4D(uvR.x, uvR.y, uv_cal.x, uv_cal.y)  :
							 new Vec2D(uv_cal.y, uv_cal.y);
			
				v_list.push(uv_c[1][j]);
				uv_list.push(uv_tol);
			}
			
		}
		if (v_list.length > 0) {
			setUVWithList(core, v_list, uv_list);
		}
		obj.update();
	}
}

